// Background Service Worker
// Monitors download progress even when popup is closed

let currentDownload = {
    isActive: false,
    tabId: null,
    config: null,
    progress: {
        status: 'idle',
        totalRecords: 0,
        currentPage: 0,
        totalPages: 0,
        timeElapsed: 0,
        recordsPerSecond: 0,
        percentage: 0
    }
};

let progressMonitorInterval = null;

// Listen for messages from popup and content script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'startDownload') {
        startDownloadMonitoring(request.tabId, request.config);
        sendResponse({ success: true });
    } else if (request.action === 'getDownloadStatus') {
        sendResponse(currentDownload);
    } else if (request.action === 'cancelDownload') {
        cancelDownload();
        sendResponse({ success: true });
    } else if (request.action === 'progressUpdate') {
        // Content script sends progress updates here
        updateProgress(request.progress);
        sendResponse({ success: true });
    }
    return true;
});

// Start monitoring a download
function startDownloadMonitoring(tabId, config) {
    currentDownload.isActive = true;
    currentDownload.tabId = tabId;
    currentDownload.config = config;
    currentDownload.progress.status = 'starting';

    // Start polling for progress
    if (progressMonitorInterval) {
        clearInterval(progressMonitorInterval);
    }

    progressMonitorInterval = setInterval(() => {
        pollProgress();
    }, 1000); // Poll every second
}

// Poll for progress from the content script
async function pollProgress() {
    if (!currentDownload.isActive || !currentDownload.tabId) {
        return;
    }

    try {
        chrome.tabs.sendMessage(currentDownload.tabId, { action: 'getProgress' }, (response) => {
            if (chrome.runtime.lastError) {
                // Tab might be closed or page navigated away
                console.log('Lost connection to tab');
                return;
            }

            if (response && response.progress) {
                updateProgress(response.progress);
            }
        });
    } catch (error) {
        console.error('Error polling progress:', error);
    }
}

// Update progress and check for completion
function updateProgress(progress) {
    currentDownload.progress = { ...currentDownload.progress, ...progress };

    // Check if completed
    if (progress.status === 'completed') {
        handleCompletion();
    } else if (progress.status === 'cancelled') {
        handleCancellation();
    } else if (progress.status === 'error') {
        handleError(progress.error);
    }
}

// Handle download completion
function handleCompletion() {
    if (progressMonitorInterval) {
        clearInterval(progressMonitorInterval);
        progressMonitorInterval = null;
    }

    // Show notification
    chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon128.png',
        title: 'NetSuite Export Complete!',
        message: `Successfully exported ${currentDownload.progress.totalRecords.toLocaleString()} records`,
        priority: 2
    });

    currentDownload.isActive = false;
}

// Handle cancellation
function handleCancellation() {
    if (progressMonitorInterval) {
        clearInterval(progressMonitorInterval);
        progressMonitorInterval = null;
    }

    chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon128.png',
        title: 'NetSuite Export Cancelled',
        message: `Export cancelled after ${currentDownload.progress.totalRecords.toLocaleString()} records`,
        priority: 1
    });

    currentDownload.isActive = false;
}

// Handle error
function handleError(errorMsg) {
    if (progressMonitorInterval) {
        clearInterval(progressMonitorInterval);
        progressMonitorInterval = null;
    }

    chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon128.png',
        title: 'NetSuite Export Error',
        message: errorMsg || 'An error occurred during export',
        priority: 2
    });

    currentDownload.isActive = false;
}

// Cancel download
function cancelDownload() {
    if (currentDownload.isActive && currentDownload.tabId) {
        chrome.tabs.sendMessage(currentDownload.tabId, { action: 'cancelDownload' });
    }

    if (progressMonitorInterval) {
        clearInterval(progressMonitorInterval);
        progressMonitorInterval = null;
    }

    currentDownload.progress.status = 'cancelled';
    currentDownload.isActive = false;
}

// Clean up when extension is updated/reloaded
chrome.runtime.onSuspend.addListener(() => {
    if (progressMonitorInterval) {
        clearInterval(progressMonitorInterval);
    }
});

console.log('NetSuite Exporter background service worker loaded');
