// Content script - runs on NetSuite pages
// Detects saved search and injects downloader script

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'detectSearch') {
        const searchId = detectSavedSearchId();
        sendResponse({ searchId: searchId });
    } else if (request.action === 'startDownload') {
        startDownload(request.config);
        sendResponse({ success: true });
    } else if (request.action === 'getProgress') {
        const progress = getDownloadProgress();
        sendResponse(progress);
    } else if (request.action === 'pauseDownload') {
        pauseDownload();
        sendResponse({ success: true });
    } else if (request.action === 'resumeDownload') {
        resumeDownload();
        sendResponse({ success: true });
    } else if (request.action === 'cancelDownload') {
        cancelDownload();
        sendResponse({ success: true });
    }

    return true; // Keep message channel open for async response
});

// Detect saved search ID from the page
function detectSavedSearchId() {
    // Check if we're on a saved search edit page
    const url = window.location.href;

    // Method 1: Check URL parameters
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.has('id')) {
        return urlParams.get('id');
    }

    // Method 2: Check if there's a search ID input field
    const idInput = document.querySelector('input[name="id"]');
    if (idInput && idInput.value) {
        return idInput.value;
    }

    // Method 3: Check page title or specific elements
    const titleElement = document.querySelector('#st');
    if (titleElement) {
        const titleText = titleElement.textContent;
        const match = titleText.match(/customsearch\d+/);
        if (match) {
            return match[0];
        }
    }

    // Method 4: Try to find it in the page's JavaScript context
    // This works because NetSuite exposes search info in the page
    try {
        // Access the page's window object
        const searchIdElement = document.evaluate(
            "//span[contains(text(), 'customsearch')]",
            document,
            null,
            XPathResult.FIRST_ORDERED_NODE_TYPE,
            null
        ).singleNodeValue;

        if (searchIdElement) {
            const match = searchIdElement.textContent.match(/customsearch[a-z0-9_]+/i);
            if (match) {
                return match[0];
            }
        }
    } catch (e) {
        console.log('Could not detect search ID via XPath');
    }

    return null;
}

// Inject and start the downloader script
function startDownload(config) {
    // Inject the downloader script
    const script = document.createElement('script');
    script.src = chrome.runtime.getURL('downloader.js');
    script.onload = function() {
        // Pass config to the injected script
        window.postMessage({
            type: 'NS_EXPORT_START',
            config: config
        }, '*');
    };
    (document.head || document.documentElement).appendChild(script);
}

// Get download progress from the page
function getDownloadProgress() {
    // The downloader script will set this on window
    if (window.nsExportProgress) {
        return { progress: window.nsExportProgress };
    }

    return {
        progress: {
            status: 'unknown',
            totalRecords: 0,
            currentPage: 0,
            totalPages: 0,
            timeElapsed: 0,
            recordsPerSecond: 0,
            percentage: 0
        }
    };
}

// Cancel download
function cancelDownload() {
    window.postMessage({ type: 'NS_EXPORT_CANCEL' }, '*');
}

// Listen for progress updates from the injected script
window.addEventListener('message', (event) => {
    if (event.source !== window) return;

    if (event.data.type === 'NS_EXPORT_PROGRESS') {
        // Store progress locally
        window.nsExportProgress = event.data.progress;

        // Send progress update to background worker
        chrome.runtime.sendMessage({
            action: 'progressUpdate',
            progress: event.data.progress
        });
    }
});
