// NetSuite Downloader Script - Injected into page context
// This script runs in the page's context to access NetSuite's N/search module

(function() {
    'use strict';

    let config = null;
    let db = null;
    let totalRecords = 0;
    let isCancelled = false;
    let startTime = null;
    let searchName = '';

    const DB_NAME = 'NetSuiteExport';
    const STORE_NAME = 'records';

    // Progress tracking
    window.nsExportProgress = {
        status: 'idle',
        totalRecords: 0,
        currentPage: 0,
        totalPages: 0,
        timeElapsed: 0,
        recordsPerSecond: 0,
        percentage: 0,
        estimatedTimeRemaining: 0
    };

    // Listen for messages from content script
    window.addEventListener('message', async (event) => {
        if (event.source !== window) return;

        if (event.data.type === 'NS_EXPORT_START') {
            config = event.data.config;
            await startExport();
        } else if (event.data.type === 'NS_EXPORT_CANCEL') {
            isCancelled = true;
            updateProgress({ status: 'cancelled' });
        }
    });

    // Update progress and post message
    function updateProgress(updates) {
        Object.assign(window.nsExportProgress, updates);

        window.postMessage({
            type: 'NS_EXPORT_PROGRESS',
            progress: window.nsExportProgress
        }, '*');
    }

    // IndexedDB functions
    async function initDB() {
        return new Promise((resolve, reject) => {
            const request = indexedDB.open(DB_NAME, 1);
            request.onerror = () => reject(request.error);
            request.onsuccess = () => resolve(request.result);
            request.onupgradeneeded = (event) => {
                const db = event.target.result;
                if (!db.objectStoreNames.contains(STORE_NAME)) {
                    const objectStore = db.createObjectStore(STORE_NAME, { keyPath: 'id', autoIncrement: true });
                    objectStore.createIndex('internalId', 'internalId', { unique: false });
                }
            };
        });
    }

    async function clearDB() {
        return new Promise((resolve, reject) => {
            // Set a timeout in case clearing hangs
            const timeout = setTimeout(() => {
                console.warn('Database clear timeout - forcing continue');
                resolve(); // Resolve anyway to continue
            }, 10000); // 10 second timeout

            // First check if there's data to clear
            const countTransaction = db.transaction([STORE_NAME], 'readonly');
            const countStore = countTransaction.objectStore(STORE_NAME);
            const countRequest = countStore.count();

            countRequest.onsuccess = () => {
                const count = countRequest.result;

                if (count === 0) {
                    // Nothing to clear
                    clearTimeout(timeout);
                    resolve();
                    return;
                }

                // There's old data - clear it
                updateProgress({ status: 'clearing_old_data' });

                const transaction = db.transaction([STORE_NAME], 'readwrite');
                const objectStore = transaction.objectStore(STORE_NAME);
                const request = objectStore.clear();
                request.onerror = () => {
                    clearTimeout(timeout);
                    reject(request.error);
                };
                request.onsuccess = () => {
                    clearTimeout(timeout);
                    resolve();
                };
            };

            countRequest.onerror = () => {
                clearTimeout(timeout);
                reject(countRequest.error);
            };
        });
    }

    async function addRecord(record) {
        return new Promise((resolve, reject) => {
            const transaction = db.transaction([STORE_NAME], 'readwrite');
            const objectStore = transaction.objectStore(STORE_NAME);
            const request = objectStore.add(record);
            request.onerror = () => {
                if (request.error.name === 'ConstraintError') {
                    resolve();
                } else {
                    reject(request.error);
                }
            };
            request.onsuccess = () => resolve();
        });
    }

    async function getAllRecords() {
        return new Promise((resolve, reject) => {
            const transaction = db.transaction([STORE_NAME], 'readonly');
            const objectStore = transaction.objectStore(STORE_NAME);
            const request = objectStore.getAll();
            request.onerror = () => reject(request.error);
            request.onsuccess = () => resolve(request.result);
        });
    }

    async function getRecordRange(offset, limit) {
        return new Promise((resolve, reject) => {
            const transaction = db.transaction([STORE_NAME], 'readonly');
            const objectStore = transaction.objectStore(STORE_NAME);
            const records = [];
            let skipped = 0;
            let collected = 0;

            const cursorRequest = objectStore.openCursor();
            cursorRequest.onerror = () => reject(cursorRequest.error);
            cursorRequest.onsuccess = (event) => {
                const cursor = event.target.result;
                if (cursor) {
                    if (skipped < offset) {
                        skipped++;
                        cursor.continue();
                    } else if (collected < limit) {
                        records.push(cursor.value);
                        collected++;
                        cursor.continue();
                    } else {
                        resolve(records);
                    }
                } else {
                    resolve(records);
                }
            };
        });
    }

    // Chunked CSV download for a specific batch of records
    async function downloadCSVChunkedBatch(offset, recordCount, filename, batchNum, totalBatches) {
        const CHUNK_SIZE = 2000;

        // Get first record to determine headers
        const firstRecords = await getRecordRange(offset, 1);
        const headers = firstRecords.length > 0 ? Object.keys(firstRecords[0]) : [];

        const totalChunks = Math.ceil(recordCount / CHUNK_SIZE);
        const batchStartTime = Date.now();

        // Create a temporary array to accumulate chunks
        let tempParts = [];
        tempParts.push(headers.map(h => `"${h}"`).join(',') + '\n');

        // Process chunks for this batch
        for (let i = 0; i < totalChunks; i++) {
            const start = offset + (i * CHUNK_SIZE);
            const chunkSize = Math.min(CHUNK_SIZE, recordCount - (i * CHUNK_SIZE));
            let records = await getRecordRange(start, chunkSize);

            // Convert records to CSV rows
            let csvRows = '';
            for (let j = 0; j < records.length; j++) {
                const record = records[j];
                const row = headers.map(header => {
                    let value = String(record[header] || '');
                    value = value.replace(/"/g, '""');
                    if (value.includes(',') || value.includes('\n') || value.includes('"')) {
                        value = `"${value}"`;
                    }
                    return value;
                });
                csvRows += row.join(',') + '\n';
            }

            tempParts.push(csvRows);

            // Calculate estimated time remaining for this batch
            const batchElapsed = Math.floor((Date.now() - batchStartTime) / 1000);
            const chunksRemaining = totalChunks - (i + 1);
            const avgTimePerChunk = batchElapsed / (i + 1);
            const batchTimeRemaining = Math.floor(chunksRemaining * avgTimePerChunk);

            // Estimate total time for remaining batches (rough estimate)
            const batchesRemaining = totalBatches - batchNum;
            const estimatedTimeRemaining = batchTimeRemaining + (batchesRemaining * batchElapsed);

            // Update progress
            const batchProgress = Math.round(((i + 1) / totalChunks) * 100);
            updateProgress({
                status: 'exporting',
                currentBatch: batchNum,
                totalBatches: totalBatches,
                batchProgress: batchProgress,
                percentage: Math.round(((batchNum - 1 + ((i + 1) / totalChunks)) / totalBatches) * 100),
                estimatedTimeRemaining: estimatedTimeRemaining
            });

            // Clear memory
            records = null;
            csvRows = null;

            // GC pause
            await new Promise(resolve => setTimeout(resolve, 10));
        }

        // Create and download the batch file
        const blob = new Blob(tempParts, { type: 'text/csv;charset=utf-8;' });
        tempParts = null;

        const link = document.createElement('a');
        link.href = URL.createObjectURL(blob);
        link.download = filename;
        link.style.display = 'none';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);

        // Cleanup
        setTimeout(() => URL.revokeObjectURL(link.href), 1000);
    }

    // Streaming CSV download using File System Access API to write directly to disk
    async function downloadCSVStreaming(filename, fileHandle) {
        try {
            // If no fileHandle provided (fallback scenario), use blob method
            if (!fileHandle) {
                await downloadCSVStreamingFallback(filename);
                return;
            }

            // Create a writable stream from the pre-selected file
            const writable = await fileHandle.createWritable();

            const CHUNK_SIZE = 5000; // Process 5000 records at a time

            // Get first record to determine headers
            const firstRecords = await getRecordRange(0, 1);
            const headers = firstRecords.length > 0 ? Object.keys(firstRecords[0]) : [];

            // Write CSV header
            await writable.write(headers.map(h => `"${h}"`).join(',') + '\n');

            const totalChunks = Math.ceil(totalRecords / CHUNK_SIZE);
            const exportStartTime = Date.now();

            // Process and write chunks directly to disk
            for (let i = 0; i < totalChunks; i++) {
                const start = i * CHUNK_SIZE;
                let records = await getRecordRange(start, Math.min(CHUNK_SIZE, totalRecords - start));

                // Convert records to CSV rows
                let csvRows = '';
                for (let j = 0; j < records.length; j++) {
                    const record = records[j];
                    const row = headers.map(header => {
                        let value = String(record[header] || '');
                        value = value.replace(/"/g, '""');
                        if (value.includes(',') || value.includes('\n') || value.includes('"')) {
                            value = `"${value}"`;
                        }
                        return value;
                    });
                    csvRows += row.join(',') + '\n';
                }

                // Write this chunk to disk
                await writable.write(csvRows);

                // Calculate estimated time remaining for export phase
                const exportElapsed = Math.floor((Date.now() - exportStartTime) / 1000);
                const chunksRemaining = totalChunks - (i + 1);
                const avgTimePerChunk = exportElapsed / (i + 1);
                const estimatedTimeRemaining = Math.floor(chunksRemaining * avgTimePerChunk);

                // Calculate total elapsed time from start
                const totalElapsed = Math.floor((Date.now() - startTime) / 1000);

                // Update progress
                const chunkProgress = Math.round(((i + 1) / totalChunks) * 100);
                updateProgress({
                    status: 'exporting',
                    percentage: chunkProgress,
                    currentChunk: i + 1,
                    totalChunks: totalChunks,
                    timeElapsed: totalElapsed,
                    estimatedTimeRemaining: estimatedTimeRemaining
                });

                // Clear memory
                records = null;
                csvRows = null;

                // Small pause for UI updates
                await new Promise(resolve => setTimeout(resolve, 0));
            }

            // Close the file
            updateProgress({ status: 'finalizing', percentage: 100, estimatedTimeRemaining: 0 });
            await writable.close();

        } catch (error) {
            // If user cancels the save dialog or File System Access API is not supported
            if (error.name === 'AbortError') {
                updateProgress({ status: 'cancelled', error: 'File save cancelled' });
            } else if (error.name === 'NotSupportedError' || !window.showSaveFilePicker) {
                // Fallback to old method for unsupported browsers
                console.warn('File System Access API not supported, falling back to blob method');
                await downloadCSVStreamingFallback(filename);
            } else {
                throw error;
            }
        }
    }

    // Fallback method using smaller blob parts (for browsers without File System Access API)
    async function downloadCSVStreamingFallback(filename) {
        const CHUNK_SIZE = 1000;
        const BLOB_THRESHOLD = 20; // Create blob every 20k records

        const firstRecords = await getRecordRange(0, 1);
        const headers = firstRecords.length > 0 ? Object.keys(firstRecords[0]) : [];
        const totalChunks = Math.ceil(totalRecords / CHUNK_SIZE);

        const blobParts = [];
        blobParts.push(new Blob([headers.map(h => `"${h}"`).join(',') + '\n'], { type: 'text/csv' }));

        let tempChunks = [];
        const exportStartTime = Date.now();

        for (let i = 0; i < totalChunks; i++) {
            const start = i * CHUNK_SIZE;
            let records = await getRecordRange(start, Math.min(CHUNK_SIZE, totalRecords - start));

            let csvRows = '';
            for (let j = 0; j < records.length; j++) {
                const record = records[j];
                const row = headers.map(header => {
                    let value = String(record[header] || '');
                    value = value.replace(/"/g, '""');
                    if (value.includes(',') || value.includes('\n') || value.includes('"')) {
                        value = `"${value}"`;
                    }
                    return value;
                });
                csvRows += row.join(',') + '\n';
            }

            tempChunks.push(csvRows);

            if (tempChunks.length >= BLOB_THRESHOLD || i === totalChunks - 1) {
                blobParts.push(new Blob(tempChunks, { type: 'text/csv' }));
                tempChunks = [];
            }

            // Calculate estimated time remaining
            const exportElapsed = Math.floor((Date.now() - exportStartTime) / 1000);
            const chunksRemaining = totalChunks - (i + 1);
            const avgTimePerChunk = exportElapsed / (i + 1);
            const estimatedTimeRemaining = Math.floor(chunksRemaining * avgTimePerChunk);

            // Calculate total elapsed time from start
            const totalElapsed = Math.floor((Date.now() - startTime) / 1000);

            const chunkProgress = Math.round(((i + 1) / totalChunks) * 100);
            updateProgress({
                status: 'exporting',
                percentage: chunkProgress,
                currentChunk: i + 1,
                totalChunks: totalChunks,
                timeElapsed: totalElapsed,
                estimatedTimeRemaining: estimatedTimeRemaining
            });

            records = null;
            csvRows = null;

            if (i % 10 === 0) {
                await new Promise(resolve => setTimeout(resolve, 15));
            }
        }

        updateProgress({ status: 'finalizing', percentage: 100, estimatedTimeRemaining: 0 });
        const finalBlob = new Blob(blobParts, { type: 'text/csv;charset=utf-8;' });
        blobParts.length = 0;

        const link = document.createElement('a');
        link.href = URL.createObjectURL(finalBlob);
        link.download = filename;
        link.style.display = 'none';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);

        setTimeout(() => URL.revokeObjectURL(link.href), 1000);
    }

    // Download CSV file
    function downloadCSV(data, filename) {
        if (data.length === 0) return;

        // Get headers from first record
        const headers = Object.keys(data[0]);
        let csv = headers.map(h => `"${h}"`).join(',') + '\n';

        data.forEach(record => {
            const row = headers.map(header => {
                let value = String(record[header] || '');
                value = value.replace(/"/g, '""');
                if (value.includes(',') || value.includes('\n') || value.includes('"')) {
                    value = `"${value}"`;
                }
                return value;
            });
            csv += row.join(',') + '\n';
        });

        const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        link.href = URL.createObjectURL(blob);
        link.download = filename;
        link.style.display = 'none';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        URL.revokeObjectURL(link.href);
    }

    // Download JSON file
    function downloadJSON(data, filename) {
        const json = JSON.stringify(data, null, 2);
        const blob = new Blob([json], { type: 'application/json;charset=utf-8;' });
        const link = document.createElement('a');
        link.href = URL.createObjectURL(blob);
        link.download = filename;
        link.style.display = 'none';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        URL.revokeObjectURL(link.href);
    }

    // Main export function
    async function startExport() {
        try {
            startTime = Date.now();

            // Request file save location FIRST (for single file mode)
            let fileHandle = null;
            if (config.downloadMode === 'single') {
                try {
                    updateProgress({ status: 'requesting_file_access' });

                    // Generate filename early
                    const tempSearchName = config.searchId.replace(/[^a-z0-9_\-\s]/gi, '_').replace(/\s+/g, '_').toLowerCase();
                    const fileExtension = config.exportFormat === 'csv' ? 'csv' : 'json';

                    fileHandle = await window.showSaveFilePicker({
                        suggestedName: `${tempSearchName}_export.${fileExtension}`,
                        types: [{
                            description: 'CSV Files',
                            accept: { 'text/csv': ['.csv'] }
                        }]
                    });
                } catch (error) {
                    if (error.name === 'AbortError') {
                        updateProgress({ status: 'cancelled', error: 'File save cancelled' });
                        return;
                    }
                    // If File System Access API not supported, fileHandle stays null and we use fallback
                    console.warn('File System Access API not available, will use fallback method');
                }
            }

            updateProgress({ status: 'initializing' });

            // Delete and recreate IndexedDB to avoid slow clearing of old data
            try {
                if (db) {
                    db.close();
                }
                await new Promise((resolve, reject) => {
                    const deleteRequest = indexedDB.deleteDatabase(DB_NAME);
                    deleteRequest.onsuccess = () => resolve();
                    deleteRequest.onerror = () => reject(deleteRequest.error);
                    deleteRequest.onblocked = () => {
                        console.warn('Database delete blocked, forcing continue');
                        setTimeout(resolve, 1000);
                    };
                });
            } catch (error) {
                console.warn('Error deleting database:', error);
            }

            // Initialize fresh IndexedDB
            db = await initDB();

            updateProgress({ status: 'loading_search' });

            // Use NetSuite's N/search module
            await new Promise((resolve, reject) => {
                require(['N/search'], function(search) {
                    (async function() {
                        try {
                            updateProgress({ status: 'search_loaded' });

                            const savedSearch = search.load({
                                id: config.searchId
                            });

                            // Debug - log available properties
                            console.log('Saved search properties:', {
                                id: savedSearch.id,
                                title: savedSearch.title,
                                name: savedSearch.name,
                                searchId: savedSearch.searchId,
                                allProps: Object.keys(savedSearch)
                            });

                            // Get the search name/title - try multiple properties
                            const searchTitle = savedSearch.title || savedSearch.name || savedSearch.id || config.searchId;
                            searchName = searchTitle;

                            // Update progress with search name
                            updateProgress({
                                status: 'search_loaded',
                                searchTitle: searchTitle
                            });

                            // Clean filename - remove invalid characters for file naming
                            const cleanSearchName = searchName.replace(/[^a-z0-9_\-\s]/gi, '_').replace(/\s+/g, '_').toLowerCase();
                            searchName = cleanSearchName;

                            const pagedData = savedSearch.runPaged({
                                pageSize: 1000
                            });

                            const totalPages = pagedData.pageRanges.length;
                            updateProgress({
                                status: 'fetching',
                                totalPages: totalPages,
                                currentPage: 0
                            });

                            // Fetch all pages
                            for (let i = 0; i < totalPages; i++) {
                                // Check for cancellation
                                if (isCancelled) {
                                    updateProgress({ status: 'cancelled' });
                                    resolve();
                                    return;
                                }

                                const pageRange = pagedData.pageRanges[i];

                                // Retry logic for network errors with exponential backoff
                                let page = null;
                                const maxRetries = 10;
                                let attempt = 0;

                                while (attempt < maxRetries && !page) {
                                    try {
                                        page = pagedData.fetch({ index: pageRange.index });
                                        break;
                                    } catch (fetchError) {
                                        attempt++;
                                        if (attempt < maxRetries) {
                                            // Exponential backoff: 2s, 4s, 8s, 16s, 32s, 60s (cap at 60s)
                                            const backoffTime = Math.min(Math.pow(2, attempt) * 1000, 60000);
                                            const secondsWait = Math.round(backoffTime / 1000);

                                            console.warn(`Network error fetching page ${i + 1}, retrying in ${secondsWait}s... (attempt ${attempt}/${maxRetries})`);
                                            updateProgress({
                                                status: 'fetching',
                                                message: `Network error, retrying in ${secondsWait}s... (attempt ${attempt}/${maxRetries})`
                                            });

                                            // Wait with backoff
                                            await new Promise(r => setTimeout(r, backoffTime));
                                        } else {
                                            throw fetchError;
                                        }
                                    }
                                }

                                // Store each record
                                for (const result of page.data) {
                                    const record = {};

                                    // Get all columns from the search result
                                    const columns = result.columns;
                                    for (let col of columns) {
                                        const fieldName = col.label || col.name;
                                        const value = result.getValue(col);
                                        const textValue = result.getText(col);

                                        // Use text value if available (for select fields), otherwise use raw value
                                        record[fieldName] = textValue || value || '';
                                    }

                                    await addRecord(record);
                                    totalRecords++;
                                }

                                // Update progress
                                const elapsed = Math.floor((Date.now() - startTime) / 1000);
                                const percentage = ((i + 1) / totalPages) * 100;
                                const recordsPerSecond = Math.floor(totalRecords / Math.max(elapsed, 1));

                                // Calculate estimated time remaining
                                const pagesRemaining = totalPages - (i + 1);
                                const avgTimePerPage = elapsed / (i + 1);
                                const estimatedTimeRemaining = Math.floor(pagesRemaining * avgTimePerPage);

                                updateProgress({
                                    status: 'fetching',
                                    totalRecords: totalRecords,
                                    currentPage: i + 1,
                                    timeElapsed: elapsed,
                                    recordsPerSecond: recordsPerSecond,
                                    percentage: percentage,
                                    estimatedTimeRemaining: estimatedTimeRemaining
                                });
                            }

                            // Export phase
                            updateProgress({ status: 'exporting', percentage: 100 });

                            // Determine download mode and format
                            const isCSV = config.exportFormat === 'csv';
                            const fileExtension = isCSV ? 'csv' : 'json';

                            if (config.downloadMode === 'single') {
                                // Single combined file - use streaming approach for large files
                                await downloadCSVStreaming(`${searchName}_export.${fileExtension}`, fileHandle);
                            } else {
                                // Split into batches
                                const batchSize = config.batchSize || 10000;
                                const numBatches = Math.ceil(totalRecords / batchSize);

                                for (let i = 0; i < numBatches; i++) {
                                    const start = i * batchSize;
                                    const batchRecordCount = Math.min(batchSize, totalRecords - start);

                                    // Update progress for current batch
                                    updateProgress({
                                        status: 'exporting',
                                        currentBatch: i + 1,
                                        totalBatches: numBatches,
                                        percentage: Math.round(((i + 1) / numBatches) * 100)
                                    });

                                    await downloadCSVChunkedBatch(start, batchRecordCount, `${searchName}_batch_${String(i + 1).padStart(3, '0')}.${fileExtension}`, i + 1, numBatches);

                                    // Small delay between downloads
                                    await new Promise(r => setTimeout(r, 500));
                                }
                            }

                            updateProgress({ status: 'completed', percentage: 100 });

                            // Clean up database after successful download
                            try {
                                await clearDB();
                                console.log('Database cleared after successful export');
                            } catch (cleanupError) {
                                console.warn('Failed to cleanup database:', cleanupError);
                                // Don't fail the overall operation if cleanup fails
                            }

                            resolve();

                        } catch (e) {
                            console.error('Export error:', e);
                            console.error('Error details:', {
                                name: e.name,
                                message: e.message,
                                stack: e.stack,
                                details: e.details || e.toString()
                            });
                            updateProgress({ status: 'error', error: e.message || e.toString() });
                            reject(e);
                        }
                    })();
                });
            });

        } catch (error) {
            console.error('Export failed:', error);
            console.error('Outer error details:', {
                name: error.name,
                message: error.message,
                stack: error.stack,
                details: error.details || error.toString()
            });
            updateProgress({ status: 'error', error: error.message || error.toString() });
        }
    }

    console.log('NetSuite Exporter downloader script loaded');
})();
