// Popup UI Controller
let currentSearchId = null;
let progressInterval = null;

// DOM Elements
const elements = {
    searchId: null,
    searchCustomId: null,
    settingsSection: null,
    progressSection: null,
    startBtn: null,
    cancelBtn: null,
    batchSizeContainer: null,
    batchSizeInput: null,
    downloadModeRadios: null,
    exportFormatRadios: null,
    progressBar: null,
    progressPercent: null,
    statusText: null,
    recordsCount: null,
    timeElapsed: null,
    timeRemaining: null,
    recordsPerSecond: null,
    messageArea: null,
    notOnSearchWarning: null
};

// Initialize popup
document.addEventListener('DOMContentLoaded', async () => {
    // Get all DOM elements
    elements.searchId = document.getElementById('searchId');
    elements.searchCustomId = document.getElementById('searchCustomId');
    elements.settingsSection = document.getElementById('settingsSection');
    elements.progressSection = document.getElementById('progressSection');
    elements.startBtn = document.getElementById('startBtn');
    elements.cancelBtn = document.getElementById('cancelBtn');
    elements.batchSizeContainer = document.getElementById('batchSizeContainer');
    elements.batchSizeInput = document.getElementById('batchSize');
    elements.downloadModeRadios = document.getElementsByName('downloadMode');
    elements.exportFormatRadios = document.getElementsByName('exportFormat');
    elements.progressBar = document.getElementById('progressBar');
    elements.progressPercent = document.getElementById('progressPercent');
    elements.statusText = document.getElementById('statusText');
    elements.recordsCount = document.getElementById('recordsCount');
    elements.timeElapsed = document.getElementById('timeElapsed');
    elements.timeRemaining = document.getElementById('timeRemaining');
    elements.recordsPerSecond = document.getElementById('recordsPerSecond');
    elements.messageArea = document.getElementById('messageArea');
    elements.notOnSearchWarning = document.getElementById('notOnSearchWarning');

    // Setup event listeners
    setupEventListeners();

    // Check for existing download in progress
    await checkExistingDownload();

    // Check if we're on a saved search page
    await detectSavedSearch();
});

// Setup event listeners
function setupEventListeners() {
    // Download mode radio buttons
    elements.downloadModeRadios.forEach(radio => {
        radio.addEventListener('change', (e) => {
            if (e.target.value === 'batches') {
                elements.batchSizeContainer.style.display = 'block';
            } else {
                elements.batchSizeContainer.style.display = 'none';
            }
        });
    });

    // Control buttons
    elements.startBtn.addEventListener('click', startDownload);
    elements.cancelBtn.addEventListener('click', cancelDownload);
}

// Check if there's an existing download in progress
async function checkExistingDownload() {
    chrome.runtime.sendMessage({ action: 'getDownloadStatus' }, (response) => {
        if (response && response.isActive) {
            // There's a download in progress
            showRunningState();
            startProgressMonitoring();
        }
    });
}

// Detect saved search from current tab
async function detectSavedSearch() {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

        // Send message to content script to detect search
        chrome.tabs.sendMessage(tab.id, { action: 'detectSearch' }, (response) => {
            if (chrome.runtime.lastError) {
                showNotOnSearchPage();
                return;
            }

            if (response && response.searchId) {
                currentSearchId = response.searchId;

                // Display the search ID with label
                elements.searchId.textContent = `Search ID: ${currentSearchId}`;

                elements.notOnSearchWarning.style.display = 'none';
                elements.settingsSection.style.display = 'block';
            } else {
                showNotOnSearchPage();
            }
        });
    } catch (error) {
        showNotOnSearchPage();
    }
}

// Show warning when not on search page
function showNotOnSearchPage() {
    elements.searchId.textContent = 'Not on saved search page';
    elements.settingsSection.style.display = 'none';
    elements.notOnSearchWarning.style.display = 'block';
    elements.startBtn.disabled = true;
}

// Start download
async function startDownload() {
    if (!currentSearchId) {
        showMessage('No saved search detected', 'error');
        return;
    }

    const downloadMode = document.querySelector('input[name="downloadMode"]:checked').value;
    const batchSize = parseInt(elements.batchSizeInput.value);
    const exportFormat = document.querySelector('input[name="exportFormat"]:checked').value;

    const config = {
        searchId: currentSearchId,
        downloadMode,
        batchSize,
        exportFormat
    };

    // Get current tab
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    // Tell background worker to start monitoring
    chrome.runtime.sendMessage({
        action: 'startDownload',
        tabId: tab.id,
        config: config
    }, (response) => {
        if (response && response.success) {
            // Tell content script to inject downloader
            chrome.tabs.sendMessage(tab.id, {
                action: 'startDownload',
                config: config
            }, (contentResponse) => {
                if (chrome.runtime.lastError) {
                    showMessage('Failed to start download: ' + chrome.runtime.lastError.message, 'error');
                    return;
                }

                if (contentResponse && contentResponse.success) {
                    showRunningState();
                    startProgressMonitoring();
                } else {
                    showMessage('Failed to start download', 'error');
                }
            });
        }
    });
}

// Show running state
function showRunningState() {
    elements.settingsSection.style.display = 'none';
    elements.progressSection.style.display = 'block';

    elements.startBtn.style.display = 'none';
    elements.cancelBtn.style.display = 'inline-block';
}

// Start monitoring progress from background worker
function startProgressMonitoring() {
    if (progressInterval) {
        clearInterval(progressInterval);
    }

    progressInterval = setInterval(async () => {
        chrome.runtime.sendMessage({ action: 'getDownloadStatus' }, (response) => {
            if (chrome.runtime.lastError || !response) {
                return;
            }

            updateProgress(response.progress);

            // Stop monitoring if not active
            if (!response.isActive) {
                clearInterval(progressInterval);
                progressInterval = null;
            }
        });
    }, 500); // Update UI every 500ms
}

// Update progress UI
function updateProgress(progress) {
    if (!progress) return;

    const { status, totalRecords, currentPage, totalPages, timeElapsed, recordsPerSecond, percentage, searchTitle, estimatedTimeRemaining } = progress;

    // Update search title if provided
    if (searchTitle && currentSearchId) {
        elements.searchId.textContent = `Search ID: ${searchTitle}`;
    }

    // Update progress bar
    elements.progressBar.style.width = (percentage || 0) + '%';
    elements.progressPercent.textContent = (percentage || 0).toFixed(1) + '%';

    // Update stats
    elements.statusText.textContent = getStatusText(status, progress);
    elements.recordsCount.textContent = (totalRecords || 0).toLocaleString();
    elements.timeElapsed.textContent = formatTime(timeElapsed || 0);
    elements.timeRemaining.textContent = estimatedTimeRemaining > 0 ? formatTime(estimatedTimeRemaining) : '--';
    elements.recordsPerSecond.textContent = `${recordsPerSecond || 0} rec/sec`;

    // Check if completed
    if (status === 'completed') {
        showCompletedState();
    } else if (status === 'cancelled') {
        showCancelledState();
    } else if (status === 'error') {
        showErrorState(progress.error);
    }
}

// Get friendly status text
function getStatusText(status, progress) {
    const statusMap = {
        'idle': 'Idle',
        'starting': 'Starting...',
        'initializing': 'Initializing database...',
        'clearing_old_data': 'Clearing old data...',
        'loading_search': 'Loading search...',
        'search_loaded': 'Search loaded',
        'fetching': progress && progress.message ? progress.message : 'Downloading records...',
        'requesting_file_access': 'Choose save location...',
        'exporting': progress && progress.currentBatch && progress.totalBatches
            ? `Creating batch ${progress.currentBatch}/${progress.totalBatches}${progress.batchProgress ? ` (${progress.batchProgress}%)` : ''}`
            : progress && progress.currentChunk && progress.totalChunks
            ? `Writing to file... (chunk ${progress.currentChunk}/${progress.totalChunks})`
            : progress && progress.message
            ? progress.message
            : 'Creating files...',
        'finalizing': 'Finalizing download...',
        'completed': 'Complete!',
        'cancelled': 'Cancelled',
        'error': 'Error'
    };
    return statusMap[status] || status;
}

// Cancel download
async function cancelDownload() {
    chrome.runtime.sendMessage({ action: 'cancelDownload' });
    showCancelledState();
}

// Show completed state
function showCompletedState() {
    if (progressInterval) {
        clearInterval(progressInterval);
        progressInterval = null;
    }

    elements.cancelBtn.style.display = 'none';
    elements.startBtn.style.display = 'inline-block';
    elements.startBtn.textContent = 'Download Complete!';
    elements.startBtn.disabled = true;
    elements.startBtn.classList.add('btn-success');

    showMessage('Download completed successfully! Check your Downloads folder.', 'success');
}

// Show cancelled state
function showCancelledState() {
    if (progressInterval) {
        clearInterval(progressInterval);
        progressInterval = null;
    }

    elements.cancelBtn.style.display = 'none';
    elements.startBtn.style.display = 'inline-block';
    elements.startBtn.textContent = 'Start Download';
    elements.startBtn.disabled = false;
    elements.startBtn.classList.remove('btn-success');

    // Reset to settings view
    setTimeout(() => {
        elements.progressSection.style.display = 'none';
        elements.settingsSection.style.display = 'block';
    }, 2000);

    showMessage('Download cancelled', 'warning');
}

// Show error state
function showErrorState(errorMsg) {
    if (progressInterval) {
        clearInterval(progressInterval);
        progressInterval = null;
    }

    elements.cancelBtn.style.display = 'none';
    elements.startBtn.style.display = 'inline-block';
    elements.startBtn.textContent = 'Retry Download';
    elements.startBtn.disabled = false;

    showMessage('Error: ' + (errorMsg || 'Unknown error'), 'error');
}

// Show message
function showMessage(message, type = 'info') {
    const messageDiv = document.createElement('div');
    messageDiv.className = `message message-${type}`;
    messageDiv.textContent = message;

    elements.messageArea.innerHTML = '';
    elements.messageArea.appendChild(messageDiv);

    // Auto-hide after 5 seconds (except for success messages)
    if (type !== 'success') {
        setTimeout(() => {
            if (elements.messageArea.contains(messageDiv)) {
                messageDiv.remove();
            }
        }, 5000);
    }
}

// Format time in seconds to readable format
function formatTime(seconds) {
    if (seconds < 60) {
        return `${seconds}s`;
    }

    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;

    if (mins < 60) {
        return `${mins}m ${secs}s`;
    }

    const hours = Math.floor(mins / 60);
    const remainingMins = mins % 60;
    return `${hours}h ${remainingMins}m ${secs}s`;
}

// Clean up interval when popup closes
window.addEventListener('unload', () => {
    if (progressInterval) {
        clearInterval(progressInterval);
    }
});
